-- database/schema.sql
CREATE DATABASE IF NOT EXISTS entregaja CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE entregaja;

DROP TABLE IF EXISTS order_status_logs;
DROP TABLE IF EXISTS order_offers;
DROP TABLE IF EXISTS orders;
DROP TABLE IF EXISTS motoboy_locations;
DROP TABLE IF EXISTS motoboy_profiles;
DROP TABLE IF EXISTS users;

CREATE TABLE users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(120) NOT NULL,
  email VARCHAR(160) NOT NULL UNIQUE,
  phone VARCHAR(30) NULL,
  password_hash VARCHAR(255) NOT NULL,
  role ENUM('cliente','motoboy','admin') NOT NULL,
  status ENUM('ativo','bloqueado') NOT NULL DEFAULT 'ativo',
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

CREATE TABLE motoboy_profiles (
  user_id INT PRIMARY KEY,
  available TINYINT(1) NOT NULL DEFAULT 0,
  rating_avg DECIMAL(3,2) NOT NULL DEFAULT 5.00,
  total_deliveries INT NOT NULL DEFAULT 0,
  CONSTRAINT fk_motoboy_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE motoboy_locations (
  motoboy_id INT PRIMARY KEY,
  lat DECIMAL(10,7) NOT NULL,
  lng DECIMAL(10,7) NOT NULL,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  CONSTRAINT fk_loc_motoboy FOREIGN KEY (motoboy_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE orders (
  id INT AUTO_INCREMENT PRIMARY KEY,
  client_id INT NOT NULL,
  motoboy_id INT NULL,
  details TEXT NOT NULL,
  price DECIMAL(10,2) NOT NULL,
  payment_method ENUM('pix','dinheiro') NOT NULL DEFAULT 'pix',
  payment_status ENUM('pendente','confirmado','contestacao') NOT NULL DEFAULT 'pendente',
  status ENUM('novo','procurando','ofertado','aceito','a_caminho_coleta','no_local_coleta','coletado','a_caminho_entrega','entregue','cancelado','expirado') NOT NULL DEFAULT 'novo',
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  accepted_at TIMESTAMP NULL,
  delivered_at TIMESTAMP NULL,
  cancel_reason VARCHAR(255) NULL,
  CONSTRAINT fk_order_client FOREIGN KEY (client_id) REFERENCES users(id) ON DELETE CASCADE,
  CONSTRAINT fk_order_motoboy FOREIGN KEY (motoboy_id) REFERENCES users(id) ON DELETE SET NULL
) ENGINE=InnoDB;

CREATE TABLE order_offers (
  id INT AUTO_INCREMENT PRIMARY KEY,
  order_id INT NOT NULL,
  motoboy_id INT NOT NULL,
  status ENUM('enviado','aceito','recusado','expirado') NOT NULL DEFAULT 'enviado',
  expires_at TIMESTAMP NOT NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  UNIQUE KEY uniq_offer (order_id, motoboy_id),
  CONSTRAINT fk_offer_order FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
  CONSTRAINT fk_offer_motoboy FOREIGN KEY (motoboy_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

CREATE TABLE order_status_logs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  order_id INT NOT NULL,
  status VARCHAR(40) NOT NULL,
  by_user_id INT NOT NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_log_order FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
  CONSTRAINT fk_log_user FOREIGN KEY (by_user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Seeds
INSERT INTO users (name, email, phone, password_hash, role) VALUES
('Admin', 'admin@entregaja.com', NULL, '$2b$12$E4g1xBoa7Os0ZeGqQWRNpupeb8CfhSi5y3RS5scBFplFghtun.WAu', 'admin'),
('Cliente', 'cliente@entregaja.com', NULL, '$2b$12$M7EEdTCKIF0ydAvhdQip0.fwnTejExzzHkwRfODeNQl/7LFu3y84C', 'cliente'),
('Motoboy', 'motoboy@entregaja.com', NULL, '$2b$12$mJxcMuDJ8KuzTgLefT72HeUF1tJHV8lOEEOUhE1CReZ3oTOcqI/8q', 'motoboy');

INSERT INTO motoboy_profiles (user_id, available) VALUES
((SELECT id FROM users WHERE email='motoboy@entregaja.com'), 0);

-- Log inicial exemplo (vazio por padrão)
